///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/**
 *	Contains a smart container.
 *	\file		IceSmartContainer.h
 *	\author		Pierre Terdiman
 *	\date		April, 13, 2001
 */
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Include Guard
#ifndef __ICESMARTCONTAINER_H__
#define __ICESMARTCONTAINER_H__

	class ICECORE_API SmartContainer : public Cell
	{
										DECLARE_PROTECTED_CLASS(SmartContainer)
		public:
		// Management

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	Add a cell to the container.
		 *	- This is a O(1) method
		 *	- The container is automatically resized if needed.
		 *	- A new reference to the cell is created. (owner is the container)
		 *	\param		cell	[in] the cell to store in the container
		 *	\see		Remove()
		 *	\return		true if success
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						bool			Add(Cell* cell);

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	Remove a cell from the container.
		 *	- This is a O(n) method
		 *	- Reference to the cell is deleted.
		 *	\param		cell	[in] the cell to remove from the container
		 *	\see		Add()
		 *	\return		true if success
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						bool			Remove(Cell* cell);

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	Remove all cells from the container.
		 *	This is a slow O(n*log n) method in theory, but since we always remove the first entry then
		 *	OnInvalidReference() returns in O(1), and the whole process actually is O(n)
		 *	\see		ForceCellsDestruction()
		 *	\return		true if success
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						bool			Reset();

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	Deletes all contained cells.
		 *	This is very different from Reset() since here we actually force contained cells to be deleted,
		 *	regardless of their current number of owners. (whereas Reset() only removes a single reference
		 *	to the cells, only leading to destruction if the container was the last owner)
		 *	Please note ICE is robust enough to allow that kind of savage destruction. It does not crash,
		 *	the kernel fixes everything as long as the relationships between owners & refs have been declared.
		 *	\param		context		[in] context parameter for the SelfDestruct() method
		 *	\param		userdata	[in] userdata parameter for the SelfDestruct() method
		 *	\see		Reset()
		 *	\return		true if success
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						bool			ForceCellsDestruction(udword context, udword userdata);

		// Stats

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to get the ram used by the container.
		 *	\return		the ram used in bytes.
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						udword			GetUsedRam()		const;
		// Data access.

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	Returns the current number of cells.
		 *	\see		GetCell()
		 *	\see		GetCells()
		 *	\return		current number of cells
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		__forceinline	udword			GetNbCells()		const	{ return mCurNbCells;	}

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	Returns ith cell.
		 *	\param		i	[in] cell index
		 *	\return		ith cell pointer
		 *	\see		GetNbCells()
		 *	\see		GetCells()
		 *	\warning	no bounds checking here!
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		__forceinline	Cell*			GetCell(udword i)	const	{ return mCells[i];		}

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	Returns the list of cells.
		 *	\return		list of contained cells
		 *	\see		GetCell()
		 *	\see		GetNbCells()
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		__forceinline	Cell**			GetCells()			const	{ return mCells;		}

		//! Access as an array
		__forceinline	Cell*&			operator[](udword i)const	{ ASSERT(i>=0 && i<mCurNbCells); return mCells[i];	}

		// Current cell

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	Returns current cell.
		 *	\return		current cell pointer
		 *	\see		SetCurrentCell()
		 *	\see		NextCurrent()
		 *	\see		PrevCurrent()
		 *	\see		RemoveCurrent()
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		__forceinline	Cell*			GetCurrentCell()	const	{ return mCurrent==INVALID_ID ? null : mCells[mCurrent];	}

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to setup current cell. Index is expected to be in range 0 - mCurNbCells.
		 *	\param		i	[in] index of current cell, or INVALID_ID
		 *	\return		true if success.
		 *	\see		GetCurrentCell()
		 *	\see		NextCurrent()
		 *	\see		PrevCurrent()
		 *	\see		RemoveCurrent()
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						bool			SetCurrentCell(udword i);

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to setup current cell. Input cell is expected to exist within the array.
		 *	\param		cell	[in] current cell, or null
		 *	\return		true if success.
		 *	\see		GetCurrentCell()
		 *	\see		NextCurrent()
		 *	\see		PrevCurrent()
		 *	\see		RemoveCurrent()
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						bool			SetCurrentCell(const Cell* cell);

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to increase current cell index.
		 *	\param		wrap	[in] true to wrap at the end of the array
		 *	\see		GetCurrentCell()
		 *	\see		SetCurrentCell()
		 *	\see		PrevCurrent()
		 *	\see		RemoveCurrent()
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						void			NextCurrent(bool wrap=false);

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to decrease current cell index.
		 *	\param		wrap	[in] true to wrap at the end of the array
		 *	\see		GetCurrentCell()
		 *	\see		SetCurrentCell()
		 *	\see		NextCurrent()
		 *	\see		RemoveCurrent()
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						void			PrevCurrent(bool wrap=false);

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to remove current cell from the array.
		 *	\return		true if success
		 *	\see		GetCurrentCell()
		 *	\see		SetCurrentCell()
		 *	\see		NextCurrent()
		 *	\see		PrevCurrent()
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						bool			RemoveCurrent();

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to find a contained cell, given its name.
		 *	\param		name	[in] wanted cell's name
		 *	\return		wanted cell, or null if not found
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						Cell*			FindCell(const String& name)	const;

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	This method is called by the kernel after a referenced cell has been dereferenced or deleted.
		 *	\param		invalidref		[in] the now invalid referenced cell
		 *	\return		true if the method has been overriden
		 *	\warning	Only called if CF_KERNEL_INVALIDREF is enabled
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual			bool			OnInvalidReference(const Cell* invalidref);

		private:
		// Resizing
						bool			Resize();
		// Data
						udword			mMaxNbCells;		//!< Maximum possible number of cells
						udword			mCurNbCells;		//!< Current number of cells
						Cell**			mCells;				//!< List of cells
						udword			mCurrent;			//!< Index of current cell
	};

#endif // __ICESMARTCONTAINER_H__
